﻿namespace Microsoft.Samples.PlanMyNight.Entities
{
    using System;
    using System.ComponentModel.DataAnnotations;
    using System.Linq;
    using System.Runtime.Serialization;
    using System.Web.Routing;
    using Microsoft.Samples.PlanMyNight.Data;

    [DataContract]
    public class NaturalSearchQuery : BaseSearchQuery
    {
        [DataMember]
        public int ActivityTypeId { get; set; }

        [DataMember]
        [CustomValidation(typeof(NaturalSearchQuery), "IsQueryValid")]
        public string Query { get; set; }

        public static ValidationResult IsQueryValid(string query, ValidationContext context)
        {
            // not empty
            if (string.IsNullOrWhiteSpace(query))
                return new ValidationResult("Please provide a search criteria.");

            // amount of keywords
            var keywords = query.Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
            if (keywords.Length < 2)
            {
                return new ValidationResult("Not enough parameters in the query.");
            }

            return ValidationResult.Success;
        }

        public bool IsActivityValid(IActivitiesRepository activitiesRepository)
        {
            // not empty
            if (string.IsNullOrWhiteSpace(this.Query))
                return false;

            // validate activityType
            var keywords = this.Query.Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
            var activityType = activitiesRepository.RetrieveActivityTypes().SingleOrDefault(a => a.Name.Equals(keywords[0], StringComparison.OrdinalIgnoreCase) || a.PluralizedName.Equals(keywords[0], StringComparison.OrdinalIgnoreCase));
            if (activityType != null)
            {
                this.ActivityTypeId = activityType.Id;
                return true;
            }

            return false;
        }

        public NaturalSearchQuery Clone()
        {
            return new NaturalSearchQuery
            {
                Type = this.Type,
                ActivityTypeId = this.ActivityTypeId,
                Page = this.Page,
                PageSize = this.PageSize,
                Query = this.Query,
                SortBy = this.SortBy
            };
        }

        public override BaseSearchQuery NewType(SearchType type)
        {
            var copy = this.Clone();
            copy.Type = type;
            return copy;
        }

        public override BaseSearchQuery NewPage(int page)
        {
            var copy = this.Clone();
            copy.Page = page;
            return copy;
        }

        public override BaseSearchQuery NewSortCriteria(SortCriteria sort)
        {
            var copy = this.Clone();
            copy.SortBy = sort;
            return copy;
        }

        public override RouteValueDictionary GetRouteValues()
        {
            var values = new RouteValueDictionary();
            values["Query"] = this.Query;
            values["SortBy"] = this.SortBy;
            values["Page"] = this.Page;
            values["Type"] = this.Type;
            return values;
        }
    }
}
